package io.intino.tara.builder.model;

import io.intino.tara.dsls.MetaIdentifiers;
import io.intino.tara.language.model.*;

import java.util.*;
import java.util.stream.Collectors;

import static io.intino.tara.builder.utils.Format.firstUpperCase;
import static io.intino.tara.language.model.Tag.*;
import static java.util.Collections.addAll;
import static java.util.Collections.unmodifiableList;

public class MogramImpl implements Mogram {
	private String file;
	private int line;
	private Mogram container;
	private String type;
	private String doc;
	private boolean sub;
	private String name;
	private String parentName;
	private Mogram parent;
	private boolean anonymous = true;
	private boolean dirty;
	private boolean virtual;
	private String hashCode;
	private String language;
	private String uid;
	private final String text;
	private final List<String> uses = new ArrayList<>();
	private final Map<Mogram, List<Rule>> components = new LinkedHashMap<>();
	private final List<Tag> flags = new ArrayList<>();
	private final List<Tag> annotations = new ArrayList<>();
	private final List<Parameter> parameters = new ArrayList<>();
	private final List<Variable> variables = new ArrayList<>();
	private final List<io.intino.tara.language.model.Facet> facets = new ArrayList<>();
	private final List<Mogram> children = new ArrayList<>();
	private List<FacetConstraint> facetConstraints;
	private List<String> context = new ArrayList<>();

	public MogramImpl(String text) {
		this.text = text;
	}

	@Override
	public String name() {
		return name;
	}

	@Override
	public void name(String name) {
		this.name = name;
		this.anonymous = false;
	}

	@Override
	public String file() {
		return file;
	}

	public void file(String file) {
		this.file = file;
	}

	@Override
	public String languageName() {
		return language;
	}

	@Override
	public void languageName(String language) {
		this.language = language;
	}

	@Override
	public int line() {
		return line;
	}

	@Override
	public void line(int line) {
		this.line = line;
	}

	@Override
	public String doc() {
		return doc;
	}

	public void doc(String doc) {
		this.doc = this.doc == null ? doc : this.doc + "\\n" + doc.trim();
	}

	@Override
	public boolean isSub() {
		return sub;
	}

	public void setSub(boolean sub) {
		this.sub = sub;
	}

	@Override
	public List<Mogram> subs() {
		List<Mogram> mograms = new ArrayList<>();
		children().stream().filter(Mogram::isSub).forEach(c -> {
			mograms.add(c);
			mograms.addAll(c.subs());
		});
		return unmodifiableList(mograms);
	}

	@Override
	public Mogram container() {
		return container;
	}

	@Override
	public List<String> uses() {
		return uses;
	}

	@Override
	public void container(Mogram container) {
		this.container = container;
	}

	@Override
	public boolean isTerminal() {
		return flags.contains(Terminal);
	}

	@Override
	public boolean isAbstract() {
		return flags.contains(Abstract) || children().stream().anyMatch(Mogram::isSub);
	}

	@Override
	public boolean isFacet() {
		return MetaIdentifiers.FACET.equals(type()) || is(Tag.Facet);
	}

	@Override
	public boolean isMetaFacet() {
		return MetaIdentifiers.META_FACET.equals(type());
	}

	public void facetConstraints(List<String> constraints) {
		this.facetConstraints = constraints.stream().map(FacetConstraint::new).collect(Collectors.toList());
	}

	public boolean is(Tag tag) {
		return flags.contains(tag);
	}

	public boolean into(Tag tag) {
		return annotations.contains(tag);
	}

	@Override
	public List<Tag> annotations() {
		return unmodifiableList(annotations);
	}

	@Override
	public List<Tag> flags() {
		return unmodifiableList(flags);
	}

	@Override
	public void addAnnotations(Tag... annotations) {
		addAll(this.annotations, annotations);
	}


	public void addFlags(Tag... flags) {
		Collections.addAll(this.flags, flags);
	}

	@Override
	public void addUses(List<String> imports) {
		this.uses.addAll(imports);
	}

	@Override
	public Mogram parent() {
		return parent;
	}

	public void setParent(Mogram parent) {
		this.parent = parent;
	}

	@Override
	public String parentName() {
		return parentName;
	}

	public void setParentName(String parentName) {
		this.parentName = parentName;
	}

	@Override
	public boolean isAnonymous() {
		return anonymous;
	}

	public MogramImpl isAnonymous(boolean anonymous) {
		this.anonymous = anonymous;
		return this;
	}

	@Override
	public String qualifiedName() {
		String containerQN = container.qualifiedName();
		String name = is(Instance) || isAnonymous() ? name() : firstUpperCase().format(name()).toString();
		return (containerQN.isEmpty() ? "" : containerQN + ".") + (name == null ? "[" + ANONYMOUS + shortType() + "]" : name);
	}

	public String layerQualifiedName() {
		String containerQn = container instanceof Model ? "" : ((MogramImpl) container).layerQualifiedName();
		String name = is(Instance) || isAnonymous() ? name() : firstUpperCase().format(name()).toString();
		return (containerQn.isEmpty() ? "" : containerQn + "$") + (name == null ? newUUID() : name);
	}

	public String layerQn() {
		String containerQn = container instanceof Model ? "" : ((MogramImpl) container).layerQn();
		String name = is(Instance) || isAnonymous() ? name() : firstUpperCase().format(name()).toString();
		if (name != null && is(Decorable)) name = "Abstract" + name;
		return (containerQn.isEmpty() ? "" : containerQn + "$") + (name == null ? newUUID() : name);
	}

	private String shortType() {
		return type.contains(".") ? type.substring(type.lastIndexOf(".") + 1) : type;
	}

	@Override
	public String type() {
		return type;
	}

	@Override
	public List<String> types() {
		List<String> types = new ArrayList<>();
		types.add(type());
		types.addAll(secondaryTypes());
		return unmodifiableList(types);
	}

	@Override
	public List<String> secondaryTypes() {
		Set<String> types = appliedFacets().stream().map(io.intino.tara.language.model.Facet::type).collect(Collectors.toSet());
		if (parent != null) types.addAll(parent.types());
		return List.copyOf(types);
	}

	@Override
	public void type(String type) {
		this.type = type;
	}

	@Override
	public void stashNodeName(String name) {
	}

	@Override
	public List<String> metaTypes() {
		return this.context;
	}

	@Override
	public void metaTypes(List<String> context) {
		this.context = new ArrayList<>(context);
	}

	@Override
	public Mogram resolve() {
		return this;
	}

	@Override
	public boolean isReference() {
		return false;
	}

	@Override
	public List<Parameter> parameters() {
		return Collections.unmodifiableList(parameters);
	}

	@Override
	public void addParameter(String name, String facet, int position, String extension, int line, int column, List<Object> values) {
		ParameterImpl parameter = new ParameterImpl(name, position, extension, values);
		parameter.facet(facet);
		parameter.file(file);
		parameter.line(line);
		parameter.column(column);
		parameter.owner(this);
		parameters.add(parameter);
	}

	public void add(Parameter parameter) {
		parameters.add(parameter);
	}

	@Override
	public List<Mogram> siblings() {
		List<Mogram> siblings = new ArrayList<>(container().components());
		siblings.remove(this);
		return unmodifiableList(siblings);
	}

	@Override
	public List<Mogram> components() {
		return unmodifiableList(new ArrayList<>(components.keySet()));
	}

	public void add(Mogram mogram, List<Rule> rules) {
		this.components.put(mogram, rules == null ? new ArrayList<>() : new ArrayList<>(rules));
	}

	@Override
	public List<Rule> rulesOf(Mogram component) {
		return this.components.get(component);
	}

	@Override
	public boolean contains(Mogram mogramContainer) {
		return mogramContainer != null && components.containsKey(mogramContainer);
	}

	@Override
	public void remove(Mogram mogram) {
		if (mogram != null) components.remove(mogram);
	}

	@Override
	public List<Variable> variables() {
		return unmodifiableList(variables);
	}

	@Override
	public void add(Variable... variables) {
		addAll(this.variables, variables);
	}

	@Override
	public void add(int pos, Variable... variables) {
		this.variables.addAll(pos, Arrays.asList(variables));
	}

	@Override
	public List<Mogram> referenceComponents() {
		return components.keySet().stream().filter(include -> include instanceof MogramReference).map(include -> (MogramReference) include).collect(Collectors.toUnmodifiableList());
	}

	@Override
	public Mogram targetOfReference() {
		return this;
	}

	@Override
	public List<Mogram> children() {
		return unmodifiableList(this.children);
	}

	@Override
	public void addChild(Mogram mogram) {
		children.add(mogram);
	}

	@Override
	public List<io.intino.tara.language.model.Facet> appliedFacets() {
		return unmodifiableList(facets);
	}

	@Override
	public void applyFacets(io.intino.tara.language.model.Facet... facets) {
		Collections.addAll(this.facets, facets);
	}

	@Override
	public String toString() {
		return type + " " + qualifiedName();
	}

	public String text() {
		return this.text;
	}

	private String newUUID() {
		if (uid == null) uid = UUID.randomUUID().toString();
		return is(Instance) ? uid : firstUpperCase().format(uid).toString();
	}

	public void absorb(MogramImpl node) {
		this.components.putAll(node.components);
		this.variables.addAll(node.variables);
		this.children.addAll(node.children);
		for (Mogram child : node.children) ((MogramImpl) child).setParent(this);
		this.annotations.addAll(node.annotations);
		this.flags.addAll(node.flags.stream().filter(t -> !t.equals(Abstract)).collect(Collectors.toList()));
		this.facets.addAll(node.facets);
		this.flags.remove(Abstract);
	}

	public boolean isDirty() {
		return dirty;
	}

	public void setDirty(boolean dirty) {
		this.dirty = dirty;
	}

	public boolean isVirtual() {
		return virtual;
	}

	public void setVirtual(boolean virtual) {
		this.virtual = virtual;
	}

	public void setHashCode(String hashCode) {
		this.hashCode = hashCode;
	}

	public String getHashCode() {
		return hashCode;
	}

	private static class FacetConstraint implements Mogram.FacetConstraint, Cloneable {
		private Mogram mogram;
		private boolean negated = false;
		private String name;

		FacetConstraint(String name) {
			this.name = name;
		}

		public String name() {
			return this.name;
		}

		public Mogram node() {
			return this.mogram;
		}

		public void node(Mogram mogram) {
			this.mogram = mogram;
		}

		@Override
		public FacetConstraint clone() throws CloneNotSupportedException {
			return (FacetConstraint) super.clone();
		}

		public String toString() {
			return "with " + node().qualifiedName();
		}
	}
}
