package io.intino.tara.builder.dependencyresolution;

import io.intino.tara.builder.model.Model;
import io.intino.tara.builder.model.MogramImpl;
import io.intino.tara.builder.model.MogramReference;
import io.intino.tara.builder.model.VariableReference;
import io.intino.tara.language.model.Mogram;
import io.intino.tara.language.model.MogramContainer;
import io.intino.tara.language.model.Primitive;

import java.io.File;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import static io.intino.tara.builder.utils.FileSystemUtils.getNameWithoutExtension;

public class ReferenceManager {

	private final Model model;

	ReferenceManager(Model model) {
		this.model = model;
	}

	public MogramImpl resolve(MogramReference reference) {
		return (MogramImpl) resolve(reference.getReference(), reference.container());
	}


	MogramImpl resolve(VariableReference variable, Mogram container) {
		Mogram result = resolve(variable.targetName(), container);
		return result instanceof MogramReference ? ((MogramReference) result).destination() : (MogramImpl) result;
	}

	Mogram resolveParameterReference(Primitive.Reference value, Mogram mogram) {
		String[] path = value.get().split("\\.");
		List<Mogram> roots = findRoots(mogram, path).stream().filter(n -> !n.equals(mogram)).collect(Collectors.toList());
		return selectFromOptions(mogram, path, roots);
	}

	Mogram resolve(String reference, Mogram mogram) {
		String[] path = reference.split("\\.");
		Collection<Mogram> roots = findRoots(mogram, path);
		return selectFromOptions(mogram, path, roots);
	}


	Mogram resolveParent(String reference, Mogram mogram) {
		return resolve(reference.split("\\."), searchPossibleRoots(mogram, reference.split("\\.")[0], true));
	}

	private Mogram selectFromOptions(Mogram mogram, String[] path, Collection<Mogram> roots) {
		return resolve(path, sortRootsByFile(roots, mogram.file()));
	}

	private Collection<Mogram> sortRootsByFile(Collection<Mogram> roots, String file) {
		List<Mogram> mograms = roots.stream().filter(node -> node.file().equals(file)).collect(Collectors.toList());
		roots.stream().filter(root -> !mograms.contains(root)).forEach(mograms::add);
		return mograms;
	}

	private Collection<Mogram> findRoots(Mogram mogram, String[] path) {
		Collection<Mogram> roots = searchPossibleRoots(mogram, path[0], false);
		if (!roots.isEmpty()) return roots;
		for (Mogram root : model.components())
			if (getNameWithoutExtension(new File(root.file()).getName()).equals(path[0])) {
				if (path.length == 1) continue;
				roots = searchPossibleRoots(root, path[1], false);
				break;
			}
		return roots;
	}

	private Mogram resolve(String[] path, Collection<Mogram> roots) {
		if (roots.isEmpty()) return null;
		if (roots.size() == 1 && path.length == 1) return roots.iterator().next();
		for (Mogram root : roots) {
			Mogram candidate = resolvePathInNode(path, root);
			if (candidate != null) return candidate;
		}
		return null;
	}

	private Mogram resolvePathInNode(String[] path, Mogram mogram) {
		Mogram reference = null;
		for (String name : path) {
			if (reference == null) {
				reference = areNamesake(mogram, name) ? mogram : null;
				continue;
			}
			final List<Mogram> components = reference.component(name);
			if (components.isEmpty() && reference.parent() != null)
				reference = reference.parent().component(name).get(0);
			else reference = components.isEmpty() ? null : components.get(0);
			if (reference == null) return null;
		}
		return reference;
	}

	private Collection<Mogram> searchPossibleRoots(Mogram mogram, String name, boolean parent) {
		Set<Mogram> set = new LinkedHashSet<>();
		namesake(name, set, mogram);
		addInContext(name, set, mogram, parent);
		addNodeSiblings(name, mogram, set);
		addRoots(name, set);
		return set;
	}

	private void addRoots(String name, Set<Mogram> set) {
		set.addAll(model.components().stream().
				filter(node -> areNamesake(node, name)).
				collect(Collectors.toList()));
	}

	private void addNodeSiblings(String identifier, Mogram container, Set<Mogram> set) {
		if (container == null) return;
		set.addAll(container.components().stream().filter(node -> areNamesake(node, identifier)).collect(Collectors.toList()));
	}

	private void addInContext(String name, Set<Mogram> set, Mogram mogram, boolean parent) {
		checkSiblings(name, set, mogram);
		Mogram container = mogram.container();
		while (container != null) {
			namesake(name, set, container);
			checkSiblings(name, set, container);
			container = container.container();
			if (parent) {
				final Mogram parentMogram = mogram.parent();
				if (parentMogram != null) collectParentComponents(name, set, container, parentMogram);
			}
		}
	}

	private void checkSiblings(String name, Set<Mogram> set, Mogram container) {
		for (Mogram sibling : container.siblings()) namesake(name, set, sibling);
	}

	private void collectParentComponents(String identifier, Set<Mogram> set, Mogram container, Mogram parent) {
		set.addAll(parent.components().stream().
				filter(sibling -> areNamesake(sibling, identifier) && !sibling.equals(container)).
				collect(Collectors.toList()));
	}

	private void namesake(String name, Set<Mogram> set, MogramContainer container) {
		if (container instanceof MogramImpl && areNamesake((Mogram) container, name)) set.add((Mogram) container);
	}

	private boolean areNamesake(Mogram mogram, String name) {
		return name.equals(mogram.name());
	}
}