package io.intino.tara.builder.dependencyresolution;

import io.intino.tara.builder.core.CompilerConfiguration.Level;
import io.intino.tara.builder.core.errorcollection.DependencyException;
import io.intino.tara.builder.model.Model;
import io.intino.tara.builder.model.MogramImpl;
import io.intino.tara.builder.model.MogramReference;
import io.intino.tara.dsls.MetaIdentifiers;
import io.intino.tara.language.model.*;
import io.intino.tara.language.model.rules.Size;
import io.intino.tara.language.model.rules.composition.MogramCustomRule;

import java.util.*;
import java.util.function.Predicate;

import static io.intino.tara.language.model.Tag.Abstract;

public class InheritanceResolver {
	private final Model model;

	public InheritanceResolver(Model model) {
		this.model = model;
	}

	public void resolve() throws DependencyException {
		mergeFragmentNodes();
		List<Mogram> mograms = new ArrayList<>(collectNodes(model, node -> !node.children().isEmpty()));
		sort(mograms);
		for (Mogram mogram : mograms) resolve(mogram);
	}

	private void resolve(Mogram mogram) throws DependencyException {
		List<Mogram> children = getChildrenSorted(mogram);
		if (!mogram.isAbstract() && !mogram.subs().isEmpty() && !mogram.flags().contains(Abstract))
			mogram.addFlags(Abstract);
		for (Mogram child : children) resolve(mogram, child);
	}

	private void resolve(Mogram mogram, Mogram child) throws DependencyException {
		resolveComponents(mogram, child);
		resolveVariables(mogram, child);
		resolveFlags(mogram, child);
		resolveAnnotations(mogram, child);
		resolveAllowedFacets(mogram, child);
		resolveAppliedFacets(mogram, child);
		resolveNodeRules(mogram, child);
		resolve(child);
	}

	private void mergeFragmentNodes() throws DependencyException {
		if (model.level() == Level.Model) return;
		for (List<Mogram> mograms : fragmentNodes().values()) merge(mograms);
	}

	private void merge(List<Mogram> mograms) throws DependencyException {
		if (mograms.size() <= 1) return;
		if (!correctParent(mograms))
			throw new DependencyException("Error merging extension elements. Parents are not homogeneous.", mograms.get(0));
		Mogram destination = mograms.get(0);
		if (destination == null) return;
		mograms.remove(destination);
		for (Mogram mogram : mograms) {
			((MogramImpl) destination).absorb((MogramImpl) mogram);
			model.remove(mogram);
		}
	}

	private boolean correctParent(List<Mogram> mograms) {
		String parent = mograms.get(0).parentName() == null ? "" : mograms.get(0).parentName();
		for (Mogram mogram : mograms)
			if (!parent.equals(mogram.parentName() == null ? "" : mogram.parentName())) return false;
		return true;
	}

	private Map<String, List<Mogram>> fragmentNodes() {
		Map<String, List<Mogram>> toMerge = new LinkedHashMap<>();
		for (Mogram mogram : model.components()) {
			if (mogram.isAnonymous()) continue;
			if (!toMerge.containsKey(name(mogram)))
				toMerge.put(name(mogram), new ArrayList<>());
			toMerge.get(name(mogram)).add(mogram);
		}
		return toMerge;
	}

	private String name(Mogram mogram) {
		return mogram.name() + (mogram.isFacet() ? MetaIdentifiers.FACET : "");
	}


	private void resolveNodeRules(Mogram parent, Mogram child) {
		List<Rule> parentRules = parent.container().rulesOf(parent);
		List<Rule> childRules = child.container().rulesOf(child);
		Size size = child.container().sizeOf(child);
		for (Rule<?> rule : parentRules)
			if (!(rule instanceof Size)) {
				if (!alreadyAdded(childRules, rule)) childRules.add(rule);
			} else if (isMoreRestrictiveThan((Size) rule, size)) {
				childRules.remove(size);
				childRules.add(rule);
			}
	}

	private boolean alreadyAdded(List<Rule> childRules, Rule<?> rule) {
		if (rule instanceof MogramCustomRule nr) {
			return childRules.stream().filter(r -> r instanceof MogramCustomRule).anyMatch(r -> ((MogramCustomRule) r).externalClass().equals((nr.externalClass())));
		}
		return false;
	}

	private boolean isMoreRestrictiveThan(Size parent, Size child) {
		return parent.min() > child.min() || parent.max() < child.max();
	}

	private void resolveAllowedFacets(Mogram parent, Mogram child) {
	}

	private void resolveAppliedFacets(Mogram parent, Mogram child) {
		parent.appliedFacets().stream().filter(facet -> !isOverridden(child, facet)).forEach(child::applyFacets);
	}

	private boolean isOverridden(Mogram child, Facet facet) {
		for (Facet childFacet : child.appliedFacets()) if (childFacet.type().equals(facet.type())) return true;
		return false;
	}

	private Collection<Mogram> collectNodes(Model model, Predicate<Mogram> condition) {
		Set<Mogram> collection = new HashSet<>();
		for (Mogram mogram : model.components()) {
			if (condition.test(mogram)) collection.add(mogram);
			collect(mogram, collection, condition);
		}
		return collection;
	}

	private void collect(Mogram mogram, Set<Mogram> collection, Predicate<Mogram> condition) {
		if (!(mogram instanceof MogramImpl)) return;
		if (condition.test(mogram)) collection.add(mogram);
		for (Mogram component : mogram.components()) collect(component, collection, condition);
	}

	private void resolveComponents(Mogram parent, Mogram child) {
		Map<Mogram, List<Rule>> nodes = new LinkedHashMap<>();
		for (Mogram component : parent.components()) {
			if (isOverridden(child, component)) continue;
			MogramReference reference = component.isReference() ? new MogramReference(((MogramReference) component).destination()) : new MogramReference((MogramImpl) component);
			addTags(component, reference);
			reference.setHas(false);
			reference.file(child.file());
			reference.line(child.line());
			reference.container(child);
			nodes.put(reference, component.container().rulesOf(component));
		}
		for (Map.Entry<Mogram, List<Rule>> entry : nodes.entrySet())
			child.add(entry.getKey(), entry.getValue());
	}

	private void addTags(Mogram component, MogramReference reference) {
		component.flags().stream().filter(tag -> !reference.flags().contains(tag) && Flags.forReference().contains(tag)).forEach(reference::addFlags);
		component.annotations().stream().filter(tag -> !reference.annotations().contains(tag)).forEach(reference::addAnnotations);
	}

	private void resolveFlags(Mogram parent, Mogram child) {
		parent.flags().stream().
				filter(tag -> !tag.equals(Abstract) && !child.flags().contains(tag)).
				forEach(child::addFlags);
	}

	private void resolveAnnotations(Mogram parent, Mogram child) {
		parent.annotations().stream().filter(tag -> !tag.equals(Abstract) && !child.annotations().contains(tag)).forEach(child::addAnnotations);
	}

	private void resolveVariables(Mogram parent, Mogram child) {
		List<Variable> variables = new ArrayList<>();
		for (Variable variable : parent.variables())
			if (isOverridden(child, variable)) {
				final Variable overridenVariable = findVariable(child, variable.name());
				if (overridenVariable != null) {
					overridenVariable.addFlags(variable.flags().toArray(new Tag[0]));
					overridenVariable.overriden(true);
				}
			} else variables.add(variable.cloneIt(child));
		child.add(0, variables.toArray(new Variable[0]));
	}

	private Variable findVariable(Mogram child, String name) {
		for (Variable variable : child.variables()) if (variable.name().equals(name)) return variable;
		return null;
	}

	private boolean isOverridden(MogramContainer child, Mogram mogram) {
		for (Mogram c : child.components())
			if (!isHasReference(c) && areNamesake(mogram, c) && c.type().equals(mogram.type())) {
				if (c instanceof MogramImpl && c.parent() == null) ((MogramImpl) c).setParent(mogram);
				return true;
			}
		return false;
	}

	private boolean areNamesake(Mogram mogram, Mogram c) {
		return c.name() != null && c.name().equals(mogram.name());
	}

	private boolean isHasReference(Mogram component) {
		return component instanceof MogramReference && ((MogramReference) component).isHas();
	}

	private boolean isOverridden(Mogram child, Variable variable) {
		return child.variables().stream().anyMatch(childVar -> childVar.name().equals(variable.name()) && childVar.type().equals(variable.type()));
	}

	private List<Mogram> getChildrenSorted(Mogram parent) {
		List<Mogram> children = new ArrayList<>(parent.children());
		sort(children);
		return children;
	}

	private void sort(List<Mogram> mograms) {
		if (mograms.isEmpty()) return;
		mograms.sort(inheritanceComparator());
		Collections.reverse(mograms);
	}

	private Comparator<Mogram> inheritanceComparator() {
		return new Comparator<>() {
			@Override
			public int compare(Mogram o1, Mogram o2) {
				return maxLevel(o1) - maxLevel(o2);
			}

			private int maxLevel(Mogram mogram) {
				List<Integer> levels = new ArrayList<>(Collections.singletonList(0));
				levels.addAll(mogram.children().stream().map(this::maxLevel).toList());
				levels.sort(Collections.reverseOrder());
				return 1 + levels.get(0);
			}

			@Override
			public boolean equals(Object obj) {
				return false;
			}

			@Override
			public int hashCode() {
				return super.hashCode();
			}
		};
	}
}
