package io.intino.tara.builder.dependencyresolution;

import io.intino.tara.Language;
import io.intino.tara.builder.core.errorcollection.DependencyException;
import io.intino.tara.builder.core.errorcollection.TaraException;
import io.intino.tara.builder.model.*;
import io.intino.tara.language.model.*;
import io.intino.tara.language.model.rules.CustomRule;
import io.intino.tara.language.model.rules.variable.ReferenceRule;
import io.intino.tara.language.model.rules.variable.VariableCustomRule;
import io.intino.tara.language.model.rules.variable.WordRule;

import java.io.File;
import java.lang.reflect.Field;
import java.util.*;
import java.util.stream.Collectors;

import static io.intino.tara.language.model.Primitive.REFERENCE;

public class DependencyResolver {
	private final File rulesDirectory;
	private final File semanticLib;
	private final File tempDirectory;
	private final Model model;
	private final ReferenceManager manager;
	private final Map<String, Class<?>> loadedRules = new HashMap<>();
	private final String workingPackage;
	private final List<DependencyException> rulesNotLoaded = new ArrayList<>();

	public DependencyResolver(Model model, String workingPackage, File rulesDirectory, File semanticLib, File tempDirectory) {
		this.model = model;
		this.workingPackage = workingPackage;
		this.rulesDirectory = rulesDirectory;
		this.semanticLib = semanticLib;
		this.tempDirectory = tempDirectory;
		this.manager = new ReferenceManager(this.model);
	}

	public void resolve() throws DependencyException {
		resolveParentReference(model);
		resolveInNodes(model);
	}

	public List<DependencyException> rulesNotLoaded() {
		return rulesNotLoaded;
	}

	private void resolveParentReference(Mogram mogram) throws DependencyException {
		if (mogram instanceof MogramReference) return;
		resolveParent(mogram);
		for (Mogram component : mogram.components())
			resolveParentReference(component);
	}

	private void resolveInNodes(Mogram mogram) throws DependencyException {
		resolveCustomRules(mogram);
		for (Mogram component : mogram.components()) resolve(component);
	}

	private void resolve(Mogram mogram) throws DependencyException {
		if (!(mogram instanceof MogramImpl)) return;
		resolveNodesReferences(mogram);
		resolveVariables(mogram);
		resolveParametersReference(mogram);
		resolveInNodes(mogram);
	}

	private void resolveCustomRules(Mogram mogram) throws DependencyException {
		if (mogram.container() == null) return;
		for (Rule rule : mogram.container().rulesOf(mogram))
			if (rule instanceof CustomRule) loadCustomRule(mogram, (CustomRule) rule);
	}

	private void resolveParametersReference(Parametrized parametrized) throws DependencyException {
		for (Parameter parameter : parametrized.parameters())
			resolveParameterValue((Mogram) parametrized, parameter);
	}

	private void resolveParameterValue(Mogram mogram, Parameter parameter) throws DependencyException {
		if (parameter.values().isEmpty() || !areReferenceValues(parameter)) return;
		List<Object> nodes = new ArrayList<>();
		for (Object value : parameter.values()) {
			Mogram reference = resolveReferenceParameter(mogram, (Primitive.Reference) value);
			if (reference != null) nodes.add(reference);
			else if (tryWithAnInstance((Primitive.Reference) value)) nodes.add(value);
		}
		if (!nodes.isEmpty()) {
			parameter.type(REFERENCE);
			parameter.substituteValues(nodes);
		}
	}

	private boolean tryWithAnInstance(Primitive.Reference value) {
		final Language language = model.language();
		if (language != null && language.instances().containsKey(value.get())) {
			value.setToInstance(true);
			value.instanceTypes(language.instances().get(value.get()).types());
			value.path(language.instances().get(value.get()).path());
			return true;
		}
		return false;
	}

	private Mogram resolveReferenceParameter(Mogram mogram, Primitive.Reference value) throws DependencyException {
		return manager.resolveParameterReference(value, mogram);
	}

	private boolean areReferenceValues(Parameter parameter) {
		return parameter.values().get(0) instanceof Primitive.Reference;
	}

	private void resolveParent(Mogram mogram) throws DependencyException {
		if (mogram.parent() == null && mogram.parentName() != null) {
			Mogram parent = manager.resolveParent(mogram.parentName(), getNodeContainer(mogram.container()));
			if (parent == null) throw new DependencyException("reject.dependency.parent.node.not.found", mogram);
			else {
				((MogramImpl) mogram).setParent(parent);
				parent.addChild(mogram);
			}
		}
	}

	private void resolveNodesReferences(Mogram mogram) throws DependencyException {
		for (Mogram mogramReference : mogram.referenceComponents()) {
			resolveNodeReference((MogramReference) mogramReference);
			resolveCustomRules(mogramReference);
		}
	}

	private void resolveNodeReference(MogramReference nodeReference) throws DependencyException {
		if (nodeReference.destination() != null) return;
		MogramImpl destination = manager.resolve(nodeReference);
		if (destination == null) throw new DependencyException("reject.dependency.reference.node.not.found", nodeReference);
		else nodeReference.destination(destination);
	}


	private void resolveVariables(Mogram container) throws DependencyException {
		for (Variable variable : container.variables()) {
			if (variable instanceof VariableReference) resolveVariable((VariableReference) variable, container);
			if (variable.rule() instanceof VariableCustomRule) loadCustomRule(variable);
		}
	}

	private void loadCustomRule(Variable variable) {
		final VariableCustomRule rule = (VariableCustomRule) variable.rule();
		final String source = rule.externalClass();
		File classFile = null;
		Class<?> aClass = null;
		try {
			if (loadedRules.containsKey(source)) aClass = loadedRules.get(source);
			else {
				classFile = CustomRuleLoader.compile(rule, workingPackage, rulesDirectory, semanticLib, tempDirectory);
				aClass = classFile != null ? CustomRuleLoader.load(rule, workingPackage, semanticLib, tempDirectory) : CustomRuleLoader.tryAsProvided(rule);
			}
		} catch (TaraException e) {
			rulesNotLoaded.add(new DependencyException("impossible.load.rule.class", variable, rule.externalClass(), e.getMessage()));
			rule.qualifiedName(CustomRuleLoader.composeQualifiedName(workingPackage, rule.externalClass()));
		}
		if (aClass == null) {
			rulesNotLoaded.add(new DependencyException("impossible.load.rule.class", variable, rule.externalClass()));
			return;
		} else {
			loadedRules.put(source, aClass);
			if (classFile != null) model.addRule(source, tempDirectory);
		}
		if (variable.type().equals(Primitive.WORD)) updateRule(variable, aClass);
		else {
			rule.setLoadedClass(aClass);
			rule.classFile(classFile);
		}
	}

	private void loadCustomRule(Mogram mogram, CustomRule rule) throws DependencyException {
		final String source = rule.externalClass();
		File classFile = null;
		Class<?> aClass;
		try {
			if (loadedRules.containsKey(source)) aClass = loadedRules.get(source);
			else {
				classFile = CustomRuleLoader.compile(rule, workingPackage, rulesDirectory, semanticLib, tempDirectory);
				aClass = classFile != null ? CustomRuleLoader.load(rule, workingPackage, semanticLib, tempDirectory) : CustomRuleLoader.tryAsProvided(rule);
			}
		} catch (TaraException e) {
			throw new DependencyException("impossible.load.rule.class", mogram, rule.externalClass(), e.getMessage().split("\n")[0]);
		}
		if (aClass != null) {
			loadedRules.put(source, aClass);
			if (classFile != null) model.addRule(source, tempDirectory);
			rule.setLoadedClass(aClass);
			rule.classFile(classFile);
		} //else throw new DependencyException("impossible.load.rule.class", mogram, rule.externalClass());

	}

	private void updateRule(Variable variable, Class<?> aClass) {
		if (aClass != null)
			variable.rule(new WordRule(collectEnums(Arrays.asList(aClass.getDeclaredFields())), aClass.getSimpleName()));
	}

	private List<String> collectEnums(List<Field> fields) {
		return fields.stream().filter(Field::isEnumConstant).map(Field::getName).collect(Collectors.toList());
	}

	private void resolveVariable(VariableReference variable, Mogram container) throws DependencyException {
		MogramImpl target = manager.resolve(variable, container);
		if (target != null) variable.setTarget(target);
		else if (!tryAsLanguageReference(variable))
			throw new DependencyException("reject.reference.variable.not.found", container, variable.targetName());
		variable.rule(createReferenceRule(variable));
		resolveVariableDefaultValue(variable, container);
	}

	private void resolveVariableDefaultValue(VariableReference variable, Mogram container) throws DependencyException {
		if (variable.values().isEmpty() || !(variable.values().get(0) instanceof Primitive.Reference)) return;
		final List<Primitive.Reference> collect = variable.values().stream().map(v -> ((Primitive.Reference) v)).collect(Collectors.toList());
		for (Primitive.Reference v : collect) {
			Mogram target = manager.resolve(v.get(), container);
			if (target == null)
				throw new DependencyException("reject.reference.variable.not.found", container, variable.targetName());
			v.reference(target);
		}
	}

	private boolean tryAsLanguageReference(VariableReference variable) {
		final Language language = model.language();
		if (language == null) return false;
		final List<String> types = language.types(variable.targetName());
		if (types != null) {
			variable.setTypeReference();
			variable.setTarget(new LanguageMogramReference(types, variable.targetName()));
			return true;
		}
		return false;
	}

	private ReferenceRule createReferenceRule(VariableReference variable) {
		return new ReferenceRule(collectTypes(variable.targetOfReference()));
	}

	private Set<String> collectTypes(Mogram mogram) {
		Set<String> set = new HashSet<>();
		if (!mogram.isAbstract()) set.add(mogram.qualifiedName());
		for (Mogram child : mogram.children())
			set.addAll(collectTypes(child));
		return set;
	}

	private Mogram getNodeContainer(MogramContainer reference) {
		MogramContainer container = reference;
		while (!(container instanceof MogramImpl)) {
			if (container.container() == null) break;
			container = container.container();
		}
		return (Mogram) container;
	}
}