package io.intino.tara.builder.core;

import io.intino.tara.builder.core.errorcollection.TaraException;
import io.intino.tara.builder.semantic.LanguageLoader;
import io.intino.tara.builder.utils.FileSystemUtils;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.logging.Handler;
import java.util.logging.Logger;
import java.util.logging.SimpleFormatter;
import java.util.logging.StreamHandler;

import static io.intino.builder.BuildConstants.GENERATION_PACKAGE;
import static java.io.File.separator;

public class CompilerConfiguration implements Cloneable {
	public static final String REPOSITORY = "repository";
	private static final Logger LOG = Logger.getGlobal();

	static {
		configureLog();
	}

	private int warningLevel;
	private String sourceEncoding;
	private String project;
	private String module;
	private File outDirectory;
	private boolean debug;
	private final Locale languageForCodeGeneration = Locale.ENGLISH;
	private final List<File> sourceDirectories = new ArrayList<>();
	private File resourcesDirectory;
	private List<Integer> excludedPhases = new ArrayList<>();
	private String groupID;
	private String artifactID;
	private String version;
	private final Library runtime = new Library();
	private final Library dslBuilder = new Library();
	private final ModelConfiguration model;
	private boolean verbose;
	private File tempDirectory;
	private File languagesRepository = new File(System.getProperty("user.home") + "/.m2" + separator + REPOSITORY);
	private boolean test;
	private String workingPackage;
	private PrintStream out = System.out;
	private File workspaceDirectory;

	public CompilerConfiguration() {
		setWarningLevel(1);
		setDebug(false);
		String encoding;
		encoding = System.getProperty("file.encoding", "UTF8");
		encoding = System.getProperty("tara.source.encoding", encoding);
		sourceEncoding(encoding);
		this.model = new ModelConfiguration();
		try {
			tempDirectory = Files.createTempDirectory("_tara_").toFile();
		} catch (IOException e) {
			LOG.log(java.util.logging.Level.SEVERE, e.getMessage(), e);
		}
	}

	public int getWarningLevel() {
		return this.warningLevel;
	}

	public void setWarningLevel(int level) {
		this.warningLevel = (level < 0) || (level > 3) ? 1 : level;
	}

	public String sourceEncoding() {
		return this.sourceEncoding;
	}

	public void sourceEncoding(String encoding) {
		sourceEncoding = encoding == null ? "UTF8" : encoding;
	}

	public File getOutDirectory() {
		return this.outDirectory;
	}

	public void setOutDirectory(File directory) {
		if (directory != null) {
			this.outDirectory = directory;
			this.outDirectory.mkdirs();
		} else
			this.outDirectory = null;
	}

	public boolean getDebug() {
		return this.debug;
	}

	public void setDebug(boolean debug) {
		this.debug = debug;
	}

	public File getTempDirectory() {
		return tempDirectory;
	}

	public String getProject() {
		return project;
	}

	public void setProject(String project) {
		this.project = project;
	}

	public File languagesRepository() {
		return languagesRepository;
	}

	public void languagesRepository(File languagesRepository) {
		this.languagesRepository = languagesRepository;
	}

	public String groupId() {
		return groupID;
	}

	public void groupId(String groupID) {
		this.groupID = groupID;
	}

	public String artifactId() {
		return artifactID;
	}

	public void artifactId(String artifactID) {
		this.artifactID = artifactID;
	}

	public String version() {
		return version;
	}

	public void version(String version) {
		this.version = version;
		this.model.outDslVersion(version);
	}

	public void workingPackage(String workingPackage) {
		this.workingPackage = workingPackage;
	}

	public String workingPackage() {
		return workingPackage == null || workingPackage.isEmpty() ? model.outDsl() : workingPackage;
	}

	public String dslGroupId() {
		return "tara.dsl";
	}

	public File resourcesDirectory() {
		return resourcesDirectory;
	}

	public void setResourcesDirectory(File resourcesDirectory) {
		this.resourcesDirectory = resourcesDirectory;
	}

	public void cleanTemp() {
		FileSystemUtils.removeDir(tempDirectory);
	}

	public String getModule() {
		return module;
	}

	public void setModule(String module) {
		this.module = module;
	}

	public Locale getLocale() {
		return languageForCodeGeneration;
	}

	public File getSemanticRulesLib() {
		return new File(io.intino.tara.Language.class.getProtectionDomain().getCodeSource().getLocation().getFile());
	}

	public void language(String name, String version) {
		model.language(new Language(name, version));
	}

	public void language(io.intino.tara.Language taraLanguage) {
		model.language(new Language(taraLanguage));
	}

	public ModelConfiguration model() {
		return model;
	}

	List<Integer> getExcludedPhases() {
		return excludedPhases;
	}

	public void setExcludedPhases(List<Integer> excludedPhases) {
		this.excludedPhases = excludedPhases;
	}

	public boolean isVerbose() {
		return verbose;
	}

	public void setVerbose(boolean verbose) {
		this.verbose = verbose;
	}

	public List<File> sourceDirectories() {
		return sourceDirectories;
	}

	public void setWorkspaceDirectory(File workspaceDirectory) {
		this.workspaceDirectory = new File(workspaceDirectory, "tara");
	}

	public File getImportsCache() {
		return new File(workspaceDirectory, module + ".json");
	}

	public File rulesDirectory() {
		for (File sourceDirectory : sourceDirectories) {
			final String rulesPackage = (workingPackage() == null ? module.toLowerCase() : workingPackage().toLowerCase().replace(".", File.separator)) + separator + "rules";
			final File file = new File(sourceDirectory, rulesPackage);
			if (file.exists()) return file;
		}
		return null;
	}

	public File functionsDirectory() {
		for (File sourceDirectory : sourceDirectories) {
			final String functionsPackage = (workingPackage() == null ? module.toLowerCase() : workingPackage().toLowerCase().replace(".", File.separator)) + separator + "functions";
			final File file = new File(sourceDirectory, functionsPackage);
			if (file.exists()) return file;
		}
		return null;
	}

	public boolean isTest() {
		return test;
	}

	public void setTest(boolean test) {
		this.test = test;
	}

	@Override
	public CompilerConfiguration clone() {
		try {
			return (CompilerConfiguration) super.clone();
		} catch (CloneNotSupportedException e) {
			LOG.info(e.getMessage());
			return null;
		}
	}

	public PrintStream out() {
		return out;
	}

	public void out(PrintStream out) {
		this.out = out;
	}

	public Library dslBuilder() {
		return dslBuilder;
	}

	public Library runtime() {
		return runtime;
	}

	public class ModelConfiguration {
		private Language language;
		private String outLanguageName;
		private String outLanguageVersion;
		private Level level;

		public Language language() {
			return language;
		}

		public void language(Language language) {
			this.language = language;
		}

		public void outDsl(String name) {
			this.outLanguageName = name;
		}

		public void outDslVersion(String version) {
			this.outLanguageVersion = version;
		}

		public String outDsl() {
			return outLanguageName;
		}

		public String outDslVersion() {
			return outLanguageVersion;
		}

		public void level(Level level) {
			this.level = level;
		}

		public Level level() {
			return level;
		}
	}

	public enum Level {
		Model, MetaModel, MetaMetaModel;

		public int compareLevelWith(Level type) {
			return type.ordinal() - this.ordinal();
		}
	}

	public class Language {
		io.intino.tara.Language language;
		String name;
		String version;
		String generationPackage;

		public Language(String name, String version) {
			this.name = name;
			this.version = version;
		}

		public Language(io.intino.tara.Language language) {
			this.language = language;
			this.name = this.language.languageName();
		}

		public io.intino.tara.Language get() {
			return language == null ? (language = loadLanguage()) : language;
		}

		public String name() {
			return language == null ? name : language.languageName();
		}

		public String version() {
			return version;
		}

		public void version(String version) {
			this.version = version;
		}

		public String generationPackage() {
			return this.generationPackage;
		}

		void generationPackage(File language) {
			if (language.isDirectory() || !language.exists()) {
				this.generationPackage = name;
			} else {
				try (JarFile jarFile = new JarFile(language)) {
					Manifest manifest = jarFile.getManifest();
					final Attributes tara = manifest.getAttributes("tara");
					this.generationPackage = tara == null ? name : tara.getValue(GENERATION_PACKAGE.replace(".", "-"));
				} catch (IOException e) {
					LOG.severe(e.getMessage());
				}
			}
		}

		private io.intino.tara.Language loadLanguage() {
			try {
				final io.intino.tara.Language language = LanguageLoader.load(name, version, languagesRepository().getAbsolutePath());
				generationPackage(LanguageLoader.getLanguagePath(name, version, languagesRepository().getAbsolutePath()));
				return language;
			} catch (NoClassDefFoundError | TaraException e) {
				LOG.info("Language " + name() + " cannot be load");
				return null;
			}
		}
	}

	private static void configureLog() {
		Logger.getGlobal().setLevel(java.util.logging.Level.INFO);
		LOG.setUseParentHandlers(false);
		for (Handler handler : LOG.getHandlers()) LOG.removeHandler(handler);
		final StreamHandler errorHandler = new StreamHandler(System.err, new SimpleFormatter());
		errorHandler.setLevel(java.util.logging.Level.WARNING);
		LOG.addHandler(errorHandler);
		final StreamHandler infoHandler = new StreamHandler(System.out, new SimpleFormatter());
		infoHandler.setLevel(java.util.logging.Level.INFO);
		LOG.addHandler(infoHandler);
	}

	public static class Library {
		private String groupId;
		private String artifactId;
		private String version;

		public String groupId() {
			return groupId;
		}

		public String artifactId() {
			return artifactId;
		}

		public String version() {
			return version;
		}

		public Library setGroupId(String groupId) {
			this.groupId = groupId;
			return this;
		}

		public Library setArtifactId(String artifactId) {
			this.artifactId = artifactId;
			return this;
		}

		public Library setVersion(String version) {
			this.version = version;
			return this;
		}
	}
}
