package io.intino.tara.builder.codegeneration.language;

import io.intino.itrules.Frame;
import io.intino.itrules.FrameBuilder;
import io.intino.itrules.FrameBuilderContext;
import io.intino.tara.Language;
import io.intino.tara.builder.codegeneration.TemplateTags;
import io.intino.tara.builder.core.CompilerConfiguration.Level;
import io.intino.tara.builder.model.Model;
import io.intino.tara.builder.model.MogramImpl;
import io.intino.tara.builder.model.MogramReference;
import io.intino.tara.builder.model.VariableReference;
import io.intino.tara.dsls.MetaIdentifiers;
import io.intino.tara.language.model.*;
import io.intino.tara.language.model.rules.Size;
import io.intino.tara.language.model.rules.composition.MogramCustomRule;
import io.intino.tara.language.semantics.Assumption;
import io.intino.tara.language.semantics.Constraint;
import io.intino.tara.language.semantics.Context;

import java.io.File;
import java.util.*;
import java.util.stream.Collectors;

import static io.intino.tara.builder.codegeneration.language.LanguageParameterAdapter.terminalParameters;
import static io.intino.tara.builder.core.CompilerConfiguration.Level.MetaModel;
import static io.intino.tara.builder.utils.Format.*;
import static io.intino.tara.language.model.Tag.*;
import static java.util.stream.Collectors.toList;

class LanguageModelAdapter implements io.intino.itrules.Adapter<Model>, TemplateTags {
	private static final String FacetSeparator = ":";
	public static final int BLOCK_SIZE = 5;
	private final Level level;
	private final String workingPackage;
	private final Set<Mogram> processed = new HashSet<>();
	private final String outDSL;
	private final Locale locale;
	private final Language language;
	private int rootNumber = 0;

	LanguageModelAdapter(String outDSL, Locale locale, Language language, Level level, String workingPackage) {
		this.outDSL = outDSL;
		this.locale = locale;
		this.language = language;
		this.level = level;
		this.workingPackage = workingPackage;
	}

	@Override
	public void adapt(Model model, FrameBuilderContext context) {
		initRoot(context);
		buildRootMograms(model, context);
		addInheritedRules(model, context);
	}

	private void initRoot(FrameBuilderContext root) {
		root.add(NAME, outDSL);
		root.add(TERMINAL, level.equals(MetaModel));
		root.add(META_LANGUAGE, language.languageName());
		root.add(LOCALE, locale.getLanguage());
	}

	private void buildRootMograms(Model model, FrameBuilderContext root) {
		FrameBuilder builder = new FrameBuilder(NODE);
		createRuleFrame(model, builder, root);
		List<Frame> frames = collectFrames(model);
		for (int i = 0; i < frames.size(); i += BLOCK_SIZE) {
			List<Frame> subList = frames.subList(i, Math.min(i + BLOCK_SIZE, frames.size()));
			final FrameBuilder rootFrame = new FrameBuilder(ROOT)
					.add("number", ++rootNumber)
					.add("language", outDSL);
			subList.forEach(r -> rootFrame.add(NODE, r));
			root.add("root", rootFrame.toFrame());
		}
	}

	private List<Frame> collectFrames(Model model) {
		List<Frame> frames = new ArrayList<>();
		model.components().forEach(n -> {
			final FrameBuilder rootFrame = new FrameBuilder(ROOT)
					.add("number", ++rootNumber)
					.add("language", outDSL);
			buildMogram(n, rootFrame);
			rootFrame.toFrame().frames(NODE).forEachRemaining(frames::add);
		});
		return frames;
	}

	private void buildMogram(Mogram mogram, FrameBuilder root) {
		if (alreadyProcessed(mogram)) return;
		FrameBuilder frame = new FrameBuilder(NODE);
		if (!mogram.isAbstract() && !mogram.isAnonymous() && !mogram.is(Instance))
			createRuleFrame(mogram, frame, root);
		else if (mogram.is(Instance) && !mogram.isAnonymous())
			root.add(NODE, createInstanceFrame(mogram));
		if (!mogram.isAnonymous()) mogram.components().stream()
				.filter(m -> !(m instanceof MogramReference))
				.forEach(m -> buildMogram(m, root));
	}

	private void createRuleFrame(Mogram mogram, FrameBuilder builder, FrameBuilderContext root) {
		builder.add(NAME, name(mogram));
		addTypes(mogram, builder);
		addConstraints(mogram, builder);
		addAssumptions(mogram, builder);
		addDoc(mogram, builder);
		root.add(NODE, builder.toFrame());
	}

	private Frame createInstanceFrame(Mogram mogram) {
		final FrameBuilder builder = new FrameBuilder(INSTANCE).add(QN, name(mogram));
		addTypes(mogram, builder);
		builder.add("path", outDSL);
		return builder.toFrame();
	}

	private void addInheritedRules(Model model, FrameBuilderContext root) {
		new LanguageInheritanceManager(root, instanceConstraints(), language, model).fill();
	}

	private List<String> instanceConstraints() {
		return language.catalog().entrySet().stream().
				filter(entry -> isInstance(entry.getValue())).
				map(Map.Entry::getKey).collect(toList());
	}

	private boolean isInstance(Context context) {
		return context.assumptions().stream().anyMatch(a -> a instanceof Assumption.Instance);
	}

	private void addDoc(Mogram mogram, FrameBuilder frame) {
		frame.add(DOC, new FrameBuilder(DOC).
				add(LAYER, findLayer(mogram)).
				add(FILE, new File(mogram.file()).getName().replace("\\", "\\\\")).
				add(LINE, mogram.line()).
				add(DOC, mogram.doc() != null ? format(mogram.doc()) : format(text(mogram))).
				toFrame());
	}

	private String text(Mogram mogram) {
		return mogram instanceof MogramImpl ? ((MogramImpl) mogram).text() : mogram.toString();
	}

	private String findLayer(Mogram mogram) {
		return mogram instanceof Model ? "" : getQn(mogram, workingPackage);
	}

	private String format(String doc) {
		return doc.replace("\"", "\\\"").replace("\n", "\\n").replace("\r", "");
	}

	private void addTypes(Mogram mogram, FrameBuilder builder) {
		if (mogram.type() == null) return;
		FrameBuilder typesFrameBuilder = new FrameBuilder(NODE_TYPE);
		Set<String> typeSet = new LinkedHashSet<>();
		typeSet.add(mogram.type());
		Collection<String> languageTypes = getLanguageTypes(mogram);
		if (languageTypes != null) typeSet.addAll(languageTypes);
		for (String type : typeSet) typesFrameBuilder.add(TYPE, type);
		if (typesFrameBuilder.slots() > 0) builder.add(NODE_TYPE, typesFrameBuilder.toFrame());
	}


	private Collection<String> getLanguageTypes(Mogram mogram) {
		return language.types(mogram.type());
	}

	private boolean alreadyProcessed(Mogram mogram) {
		return !processed.add(mogram);
	}

	private void addConstraints(Mogram mogram, FrameBuilder builder) {
		FrameBuilder constraints = buildComponentConstraints(mogram);
		addTerminalConstrains(mogram, constraints);
		addContextConstraints(mogram, constraints);
		builder.add(CONSTRAINTS, constraints.toFrame());
	}

	private void addContextConstraints(Mogram mogram, FrameBuilder constraints) {
		if (mogram instanceof MogramImpl) {
			if (!mogram.isTerminal()) addRequiredVariableRedefines(constraints, mogram);
			addParameterConstraints(mogram.variables(), mogram.type().startsWith(MetaIdentifiers.FACET) ? mogram.name() : "", constraints,
					terminalParameters(language, mogram) + terminalParameterIndex(constraints.toFrame()));
		}
		addMetaFacetConstraints(mogram, constraints);
		addFacetConstraints(mogram, constraints);
	}

	private int terminalParameterIndex(Frame constraints) {
		final Iterator<Frame> iterator = constraints.frames(CONSTRAINT);
		int index = 0;
		while (iterator.hasNext()) if (iterator.next().is(PARAMETER)) index++;
		return index;
	}

	private void addParameterConstraints(List<Variable> variables, String facet, FrameBuilder constrainsFrame, int parentIndex) {
		int privateVariables = 0;
		for (int index = 0; index < variables.size(); index++) {
			Variable variable = variables.get(index);
			if (!variable.isPrivate() && !finalWithValues(variable))
				new LanguageParameterAdapter(language, workingPackage, level).addParameterConstraint(constrainsFrame, facet, parentIndex + index - privateVariables, variable, CONSTRAINT);
			else privateVariables++;
		}
	}

	private boolean finalWithValues(Variable variable) {
		return variable.isFinal() && !variable.values().isEmpty();
	}

	private void addMetaFacetConstraints(Mogram mogram, FrameBuilder constraints) {
		mogram.components().stream().filter(Mogram::isMetaFacet).forEach(facetMogram -> {
			List<Mogram.FacetConstraint> with = facetMogram.facetConstraints();
			FrameBuilder builder = new FrameBuilder(CONSTRAINT, META_FACET).add(VALUE, facetMogram.qualifiedName());
			if (with != null && !with.isEmpty())
				builder.add(WITH, with.stream().map(c -> c.node().qualifiedName()).toArray(Object[]::new));
			constraints.add(CONSTRAINT, builder.toFrame());
		});
	}

	private void addFacetConstraints(Mogram mogram, FrameBuilder constraintsBuilder) {
		mogram.components().stream().filter(Mogram::isFacet).forEach(facetMogram -> {
			if (facetMogram.isAbstract()) return;
			if (facetMogram.isReference()) facetMogram = facetMogram.targetOfReference();
			FrameBuilder builder = new FrameBuilder(CONSTRAINT, FACET).add(VALUE, facetMogram.qualifiedName());
			builder.add(TERMINAL, String.valueOf(facetMogram.isTerminal()));
			if (facetMogram.facetConstraints() != null && !facetMogram.facetConstraints().isEmpty())
				for (Mogram.FacetConstraint constraint : facetMogram.facetConstraints())
					builder.add(WITH, constraint.node().name());
			if (facetMogram.flags().contains(Required)) builder.add("required", "true");
			addParameterConstraints(facetMogram.variables(), facetMogram.name(), builder, 0);
			addComponentsConstraints(builder, facetMogram);
			addTerminalConstrains(facetMogram, builder);
			constraintsBuilder.add(CONSTRAINT, builder.toFrame());
		});
		addTerminalFacets(mogram, constraintsBuilder);
	}

	private void addTerminalFacets(Mogram mogram, FrameBuilder context) {
		final List<Constraint> facetAllows = language.constraints(mogram.type()).stream().filter(allow -> allow instanceof Constraint.Facet && ((Constraint.Facet) allow).terminal()).collect(toList());
		new TerminalConstraintManager(language, mogram).addConstraints(facetAllows, context);
	}

	private void addTerminalConstrains(Mogram container, FrameBuilder frame) {
		final List<Constraint> constraints = language.constraints(container.type());
		List<Constraint> terminalConstraints = constraints.stream().
				filter(c -> validComponent(container, c) || validParameter(container, c)).
				collect(toList());
		new TerminalConstraintManager(language, container).addConstraints(terminalConstraints, frame);
	}

	private boolean validParameter(Mogram container, Constraint c) {
		if (!(c instanceof Constraint.Parameter)) return false;
		return ((Constraint.Parameter) c).flags().contains(Tag.Terminal) && !isRedefined((Constraint.Parameter) c, container.variables());
	}

	private boolean validComponent(Mogram container, Constraint c) {
		if (!(c instanceof Constraint.Component)) return false;
		return is(annotations(c), Instance) && !sizeComplete(container, typeOf(c));
	}

	private boolean isRedefined(Constraint.Parameter allow, List<? extends Variable> variables) {
		for (Variable variable : variables) if (variable.name().equals(allow.name())) return true;
		return false;
	}

	private String typeOf(Constraint constraint) {
		return ((Constraint.Component) constraint).type();
	}

	private boolean sizeComplete(MogramContainer container, String type) {
		final List<Mogram> components = container.components().stream().filter(node -> node.type().equals(type)).toList();
		return !components.isEmpty() && container.sizeOf(components.get(0)).max() == components.size();
	}

	private void addRequiredVariableRedefines(FrameBuilder constraints, Mogram mogram) {
		mogram.variables().stream().
				filter(variable -> variable.isTerminal() && variable instanceof VariableReference && !((VariableReference) variable).getTarget().isTerminal()).
				forEach(variable -> constraints.add(CONSTRAINT, new FrameBuilder("redefine", CONSTRAINT).add(NAME, variable.name()).add("supertype", variable.type()).toFrame()));
	}

	private void addAssumptions(Mogram mogram, FrameBuilder frame) {
		FrameBuilder assumptions = buildAssumptions(mogram);
		if (assumptions.slots() != 0) frame.add(ASSUMPTIONS, assumptions.toFrame());
	}

	private FrameBuilder buildAssumptions(Mogram mogram) {
		FrameBuilder assumptions = new FrameBuilder(ASSUMPTIONS);
		assumptions.add(ASSUMPTION, new FrameBuilder("stashNodeName").add("value", name(mogram, workingPackage)));
		addAnnotationAssumptions(mogram, assumptions);
		return assumptions;
	}

	public static String name(Mogram owner, String workingPackage) {
		return owner instanceof Model ? "" : withDollar().format(noPackage().format(getQn(owner, workingPackage))).toString();
	}

	public static String getQn(Mogram mogram, String workingPackage) {
		return workingPackage.toLowerCase() + DOT + qualifiedName().format(layerQn(mogram)).toString();
	}

	private static String layerQn(Mogram mogram) {
		return mogram instanceof MogramReference ? ((MogramReference) mogram).layerQualifiedName() : ((MogramImpl) mogram).layerQualifiedName();
	}


	private void addAnnotationAssumptions(Mogram mogram, FrameBuilder assumptions) {
		mogram.annotations().forEach(tag -> assumptions.add(ASSUMPTION, tag.name().toLowerCase()));
		for (Tag tag : mogram.flags()) {
			if (tag.equals(Tag.Terminal)) assumptions.add(ASSUMPTION, Instance.name());
			else if (tag.equals(Feature)) assumptions.add(ASSUMPTION, Feature.name());
			else if (tag.equals(Tag.Component)) assumptions.add(ASSUMPTION, capitalize(Tag.Component.name()));
			else if (tag.equals(Tag.Volatile)) assumptions.add(ASSUMPTION, capitalize(Tag.Volatile.name()));
		}
		if (mogram.type().startsWith(MetaIdentifiers.META_FACET)) assumptions.add(ASSUMPTION, Facet.name());
		if (mogram.isFacet()) assumptions.add(ASSUMPTION, Terminal);
	}

	private FrameBuilder buildComponentConstraints(Mogram container) {
		FrameBuilder constraints = new FrameBuilder(CONSTRAINTS);
		addComponentsConstraints(constraints, container);
		return constraints;
	}

	private void addComponentsConstraints(FrameBuilder constraints, Mogram container) {
		List<Frame> frames = new ArrayList<>();
		createComponentsConstraints(container, frames);
		frames.forEach(frame -> constraints.add(CONSTRAINT, frame));
	}

	private void createComponentsConstraints(Mogram mogram, List<Frame> frames) {
		mogram.components().stream().
				filter(c -> componentCompliant(mogram, c)).
				forEach(c -> {
					if (c.isMetaFacet()) createMetaFacetComponentConstraint(frames, c);
					else if (!c.isSub() || c.container() instanceof Model) createComponentConstraint(frames, c);
				});
	}

	private boolean componentCompliant(Mogram container, Mogram mogram) {
		return !mogram.isFacet() && (!(container instanceof MogramRoot) || rootCompliant(mogram));
	}

	private boolean rootCompliant(Mogram c) {
		return !c.is(Component) && !c.is(Feature) && !(c.isTerminal() && (c.into(Component) || c.into(Feature)));
	}


	private void createMetaFacetComponentConstraint(List<Frame> frames, Mogram mogram) {
		if (!mogram.isMetaFacet() || mogram.isAbstract()) return;
		final Mogram target = mogram.container();
		if (target.isAbstract())
			for (Mogram child : target.children()) {
				FrameBuilder builder = new FrameBuilder(CONSTRAINT, COMPONENT).add(TYPE, mogram.name() + FacetSeparator + child.qualifiedName());
				builder.add(SIZE, mogram.isTerminal() && MetaModel.compareLevelWith(level) > 0 ? transformSizeRuleOfTerminalNode(mogram) : createRulesFrames(mogram.container().rulesOf(mogram)));
				addTags(mogram, builder);
				frames.add(builder.toFrame());
			}
		else createComponentConstraint(frames, mogram);

	}

	private void createComponentConstraint(List<Frame> frames, Mogram mogram) {
		final List<Mogram> candidates = collectCandidates(mogram);
		final Size size = mogram.container().sizeOf(mogram);
		final List<Rule> allRules = mogram.container().rulesOf(mogram).stream().distinct().collect(toList());
		if ((size.isSingle() || size.isRequired() || mogram.isReference()) && candidates.size() > 1) {
			final FrameBuilder oneOfBuilder = createOneOf(candidates, allRules);
			if (!mogram.isAbstract() && !candidates.contains(mogram))
				oneOfBuilder.add(CONSTRAINT, createComponentConstraint(mogram, allRules));
			if (!mogram.isSub()) frames.add(oneOfBuilder.toFrame());
		} else {
			frames.addAll(candidates.stream().filter(c -> componentCompliant(c.container(), c)).
					map(c -> createComponentConstraint(c, allRules)).toList());
		}
	}

	private Frame createComponentConstraint(Mogram component, List<Rule> rules) {
		FrameBuilder builder = new FrameBuilder(CONSTRAINT, COMPONENT).add(TYPE, name(component));
		if (isTerminal(component)) builder.add(SIZE, transformSizeRuleOfTerminalNode(component));
		else builder.add(SIZE, createRulesFrames(rules));
		addTags(component, builder);
		return builder.toFrame();
	}

	private boolean isTerminal(Mogram component) {
		return component.isTerminal() && !isInTerminal(component) && MetaModel.compareLevelWith(level) > 0;
	}

	private String name(Mogram mogram) {
		return mogram instanceof MogramReference ?
				((MogramReference) mogram).destination().qualifiedName() :
				mogram.qualifiedName();
	}

	private FrameBuilder createOneOf(Collection<Mogram> candidates, List<Rule> rules) {
		FrameBuilder builder = new FrameBuilder(ONE_OF, CONSTRAINT);
		builder.add(RULE, createRulesFrames(rules));
		for (Mogram candidate : candidates)
			builder.add(CONSTRAINT, createComponentConstraint(candidate, candidate.container().rulesOf(candidate)));
		return builder;
	}

	private Frame[] createRulesFrames(List<Rule> rules) {
		return rules.stream()
				.map(rule -> rule instanceof MogramCustomRule ?
						buildCustomRuleFrame((MogramCustomRule) rule) :
						new FrameBuilder().append(rule).toFrame())
				.filter(Objects::nonNull)
				.toArray(Frame[]::new);
	}

	private Frame buildCustomRuleFrame(MogramCustomRule rule) {
		if (rule.loadedClass() == null) return null;
		return new FrameBuilder("rule", "customRule")
				.add("qn", rule.loadedClass().getName())
				.toFrame();
	}

	private boolean isInTerminal(Mogram component) {
		return component.container().isTerminal();
	}

	private Frame transformSizeRuleOfTerminalNode(Mogram component) {
		final Size rule = component.container().sizeOf(component);
		final Size size = new Size(0, rule.max(), rule);
		return new FrameBuilder().append(size).toFrame();
	}

	private void addTags(Mogram mogram, FrameBuilder frame) {
		Set<String> tags = mogram.annotations().stream().map(Tag::name).collect(Collectors.toCollection(LinkedHashSet::new));
		mogram.flags().stream().filter(f -> !f.equals(Decorable) && !Required.equals(f)).forEach(tag -> tags.add(convertTag(tag)));
		frame.add(TAGS, tags.toArray(new Object[0]));
	}

	private List<Mogram> collectCandidates(Mogram mogram) {
		Set<Mogram> mograms = new LinkedHashSet<>();
		if (mogram.isAnonymous() || mogram.is(Instance)) return new ArrayList<>(mograms);
		if (!mogram.isAbstract()) mograms.add(mogram);
		getNonAbstractChildren(mogram, mograms);
		return new ArrayList<>(mograms);
	}

	private void getNonAbstractChildren(Mogram mogram, Set<Mogram> mograms) {
		for (Mogram child : mogram.children())
			if (child.isAbstract())
				getNonAbstractChildren(child, mograms);
			else if (child.container().equals(mogram.container()) || mogram.isReference()) mograms.add(child);
	}

	private String convertTag(Tag tag) {
		if (tag.equals(Tag.Terminal)) return Instance.name();
		return tag.name();
	}

	private List<Tag> annotations(Constraint constraint) {
		return ((Constraint.Component) constraint).annotations();
	}

	private boolean is(List<Tag> annotations, Tag tag) {
		return annotations.contains(tag);
	}
}