package io.intino.tara.builder;

import io.intino.tara.builder.core.CompilerConfiguration;
import io.intino.tara.builder.core.CompilerConfiguration.Level;

import java.io.File;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static io.intino.builder.BuildConstants.ENCODING;
import static io.intino.builder.BuildConstants.LEVEL;

public class Tarac {
	public static void main(String[] args) {
		if (args[0].equals("--help")) {
			System.out.println(help());
			return;
		}
		TaraCompilerRunner runner = new TaraCompilerRunner(false, Collections.emptyList());
		runner.run(loadConfiguration(args), sourcesFrom(args));
	}

	private static String help() {
		StringBuilder builder = new StringBuilder();
		builder.append("Usage: tarac <options> <source files>\nwhere possible options include:\n");
		builder.append("  --level <level>         		REQUIRED. Specify the abstraction level in MDE: model (1), meta-model (2), meta-meta-model (3)\n");
		builder.append("  --dsl <name:version> 			REQUIRED. Specify the name and version of the language used for define models\n");
		builder.append("  --out-dsl <name:version>      REQUIRED. Specify the name and version of the language created. Not necessary if level is model\n");
		builder.append("  --dsl-repository <directory>  Specify the directory where the where the language will be found and out language will be created. Used '~/.m2/repository' as default\n");
		builder.append("  --encoding <encoding>         Specify character encoding used by source files. UTF-8 is used by default\n");
		return builder.toString();
	}

	private static List<File> sourcesFrom(String[] args) {
		return Arrays.stream(args)
				.dropWhile(a -> a.startsWith("-"))
				.map(File::new)
				.collect(Collectors.toList());
	}

	private static CompilerConfiguration loadConfiguration(String[] args) {
		Map<String, String> arguments = Arrays.stream(args)
				.takeWhile(a -> a.startsWith("-"))
				.map(a -> a.split(" "))
				.collect(Collectors.toMap(a -> a[0].replace("-", ""), a -> a[1]));
		CompilerConfiguration conf = new CompilerConfiguration();
		arguments.keySet().forEach(name -> processArg(conf, name, arguments.get(name)));
		return conf;
	}

	private static void processArg(CompilerConfiguration conf, String name, String value) {
		switch (name) {
			case "dslrepository":
				conf.languagesRepository(new File(value));
				break;
			case "outdsl":
				final String[] dsl = value.trim().split(":");
				conf.model().outDsl(dsl[0]);
				conf.version(dsl[1]);
				if (conf.getModule() == null) conf.setModule(dsl[0]);
				if (conf.workingPackage() == null) conf.workingPackage(dsl[0]);
				break;
			case ENCODING:
				conf.sourceEncoding(value);
				break;
			case LEVEL:
				conf.model().level(Level.values()[Integer.parseInt(value) - 1]);
				break;
			case "dsl":
				final String[] aDsl = value.trim().split(":");
				conf.language(aDsl[0], aDsl[1]);
				break;
			default:
				break;
		}
	}
}
